/**
* \file: gstpipeline.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Abstract base class for encoders for OSG
*
* \component: osgstream
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <iostream>
#include <sstream>

#include "gstpipeline.h"

namespace osgGstpipeline {

gstpipeline::gstpipeline(gstpipeline::appsrcTraits &appsrc_traits)
            : context(0)
            , loop(0)
            , pipeline(NULL)
            , appsrc(NULL)
            , callbackId(0)
            , playing(false)
            , thread(0)
            , bus()
            , stop_push(false)
            , blockMode(true)
            , use_oldbuffer(false)
            , oldBuffer(NULL)
            , currentBuffer(NULL)
            , nodata_to_push(false)
{
    create(appsrc_traits);
}

gstpipeline::~gstpipeline()
{
}

void gstpipeline::create(gstpipeline::appsrcTraits &appsrc_traits)
{
//    setenv("GST_DEBUG_DUMP_DOT_DIR","/tmp/", TRUE);

    gst_init (0, 0);

    context = g_main_context_new();
    g_main_context_push_thread_default(context);
    loop = g_main_loop_new(context, FALSE);

    setAppsrcTraits(appsrc_traits);


    FILE * pFile;
    long lSize;
    char * pipeline_str = NULL;
    size_t result;
    pFile = fopen ( "/etc/pipeline.cfg" , "rb" );
    if (pFile==NULL)
    {
        std::cout<<"File open error"<<std::endl;
        exit (1);
    }

    // obtain file size:
    fseek (pFile , 0 , SEEK_END);
    lSize = ftell (pFile);
    rewind (pFile);

    // allocate memory to contain the whole file:
    pipeline_str = (char*) calloc (1, sizeof(char)*(lSize+1));
    if (pipeline_str == NULL)
    {
        std::cout <<"Memory error"<<std::endl;
        exit (2);
    }

    // copy the file into the buffer:
    result = fread (pipeline_str,1,lSize,pFile);
    if (result != lSize)
    {
        std::cout<<"Reading error"<<std::endl;
        exit (3);
    }

      // terminate
    fclose (pFile);

    if (pipeline_str)
    {
        std::cout << " Using Gstparse " << std::endl;
        std::cout << " pipeline = "<<pipeline_str<<std::endl;
        pipeline = gst_parse_launch(pipeline_str,NULL);
        g_assert (pipeline);
        free (pipeline_str);

        bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
        g_assert(bus);

        /* add watch for messages */
        gst_bus_add_signal_watch(bus);
        callbackId = g_signal_connect(bus, "message",
                                          G_CALLBACK(gstpipeline::bus_signal), this);


        appsrc = gst_bin_get_by_name (GST_BIN(pipeline), "myappsrc");
        g_assert(appsrc);
        g_assert(GST_IS_APP_SRC(appsrc));
        g_signal_connect(G_OBJECT(appsrc), "need-data", G_CALLBACK(gstpipeline::on_need_data), this);

        GstCaps    *caps = create_appsrc_caps();
        g_object_set (G_OBJECT(appsrc), "caps", caps, NULL);
        gst_caps_unref(caps);
    }
    else
    {
        std::cout << "Unable to open file";
    }

//    GST_DEBUG_BIN_TO_DOT_FILE(GST_BIN(pipeline), GST_DEBUG_GRAPH_SHOW_ALL, "pipeline");
}

void gstpipeline::setAppsrcTraits(gstpipeline::appsrcTraits &traits)
{
    appsrc_traits = traits;
}

gstpipeline::appsrcTraits &gstpipeline::getAppsrcTraits()
{
    return appsrc_traits;
}


GstCaps* gstpipeline::create_appsrc_caps()
{

    const appsrcTraits &appsrc_traits = getAppsrcTraits();
#if GST_CHECK_VERSION(1,0,0)
    return gst_caps_new_simple ("video/x-raw",
        "format", G_TYPE_STRING, "RGB16",
        "bpp",G_TYPE_INT,appsrc_traits.depth,
        "width", G_TYPE_INT, appsrc_traits.width,
        "height", G_TYPE_INT, appsrc_traits.height,
        "framerate", GST_TYPE_FRACTION, appsrc_traits.fpsNum,appsrc_traits.fpsDem,
        "depth", G_TYPE_INT, appsrc_traits.depth,
        "bpp", G_TYPE_INT, appsrc_traits.depth,
        "endianness", G_TYPE_INT, appsrc_traits.endianness,
        "red_mask", G_TYPE_INT, appsrc_traits.red_mask,
        "green_mask", G_TYPE_INT, appsrc_traits.green_mask,
        "blue_mask", G_TYPE_INT, appsrc_traits.blue_mask,
        "alpha_mask", G_TYPE_INT, appsrc_traits.alpha_mask,
        NULL);
#else
    return gst_caps_new_simple ("video/x-raw-rgb",
          "width", G_TYPE_INT, appsrc_traits.width,
          "height", G_TYPE_INT, appsrc_traits.height,
          "framerate", GST_TYPE_FRACTION, appsrc_traits.fpsNum, appsrc_traits.fpsDem,
          "depth", G_TYPE_INT, appsrc_traits.depth,
          "bpp", G_TYPE_INT, appsrc_traits.depth,
          "endianness", G_TYPE_INT, appsrc_traits.endianness,
          "red_mask", G_TYPE_INT, appsrc_traits.red_mask,
          "green_mask", G_TYPE_INT, appsrc_traits.green_mask,
          "blue_mask", G_TYPE_INT, appsrc_traits.blue_mask,
          "alpha_mask", G_TYPE_INT, appsrc_traits.alpha_mask,
          NULL);
#endif
}

void gstpipeline::set_stream_size( uint width, uint height)
{
    appsrcTraits &appsrc_traits = getAppsrcTraits();

    appsrc_traits.width = width;
    appsrc_traits.height = height;

    setAppsrcTraits(appsrc_traits);

    GstBuffer *gst_buffer;
    pthread_mutex_lock(&sync_lock);

#if !GST_CHECK_VERSION(1,0,0)
    gst_element_send_event(GST_ELEMENT (pipeline), gst_event_new_flush_start());
    gst_element_send_event(GST_ELEMENT (pipeline), gst_event_new_flush_stop());
#endif

    while(!bufferqueue.empty())
    {
        gst_buffer = bufferqueue.front().curr_buffer;
        bufferqueue.pop();
        gst_buffer_unref(gst_buffer);
    }

    GstCaps    *caps = create_appsrc_caps();

    g_object_set (G_OBJECT(appsrc), "caps", caps, NULL);
    gst_caps_unref(caps);

    pthread_mutex_unlock(&sync_lock);
}

void* gstpipeline::run(void *data)
{
    pthread_setname_np(pthread_self(), "gst_pipeline");

    gstpipeline *_pipeline = static_cast<gstpipeline *>(data);

    g_main_loop_run(_pipeline->loop);

    pthread_exit(0);
}

void gstpipeline::start()
{
    play();
    pthread_mutex_init(&sync_lock, NULL);
    pthread_cond_init(&sync_cond, NULL);

    int status;
    status = pthread_create(&thread, NULL, gstpipeline::run, (void*)this);
    if(0 != status){
        fprintf(stderr, "Failed to create thread: %d \n", status);
    }
}

gboolean gstpipeline::bus_signal(GstBus *bus, GstMessage *message, gpointer user_data)
{
    gstpipeline *_pipeline = static_cast<gstpipeline *>(user_data);
    _pipeline->onBusSignal(bus, message);
}

gboolean gstpipeline::onBusSignal(GstBus *, GstMessage *message)
{
    if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR)
    {
        GST_DEBUG_BIN_TO_DOT_FILE_WITH_TS(GST_BIN(pipeline),
                                          GST_DEBUG_GRAPH_SHOW_ALL,
                                          "osgStream-error");
    }
    else if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_STATE_CHANGED)
    {
        if (GST_MESSAGE_SRC(message) == GST_OBJECT(pipeline))
        {
            GstState oldState, newState;
            gst_message_parse_state_changed(message, &oldState, &newState, NULL);
            std::stringstream oss("osgStream-");
            oss << gst_element_state_get_name(oldState)
                << "-"
                << gst_element_state_get_name(newState);

            GST_DEBUG_BIN_TO_DOT_FILE_WITH_TS(GST_BIN(pipeline),
                                              GST_DEBUG_GRAPH_SHOW_ALL,
                                              oss.str().c_str());
            playing = newState >= GST_STATE_PAUSED;
        }
    }
}

void gstpipeline::on_need_data(GstAppSrc *src, guint length, gpointer user_data)
{
    gstpipeline *_appsrc = static_cast<gstpipeline *>(user_data);
    _appsrc->onNeedData(src);
}

void gstpipeline::onNeedData(GstAppSrc *)
{
    if(oldBuffer != NULL)
    {
        pthread_mutex_lock(&sync_lock);
        if(!bufferqueue.empty())
        {
            if(!blockMode)
            {
                GstBuffer *latest_buffer = NULL;
                while(!(bufferqueue.empty()))
                {
                    if(latest_buffer)
                    {
                        gst_buffer_unref(latest_buffer);
                        std::cout << " func "<<__func__<<" line " << __LINE__<<"skipping buffer  "<<latest_buffer<< std::endl;
                    }

                    latest_buffer = bufferqueue.front().curr_buffer;
                    bufferqueue.pop();
                }

                if(latest_buffer != NULL)
                {
                    push_appsrc(latest_buffer);
                }
            }
            else
            {
                currentBuffer = bufferqueue.front().curr_buffer;
                bufferqueue.pop();
                push_appsrc(currentBuffer);
            }
        }
        else
        {
            if(use_oldbuffer)
            {
                /* pushing old buffer again */
                push_appsrc(oldBuffer);
            }
            else
            {
                nodata_to_push = true;
            }
        }
        pthread_mutex_unlock(&sync_lock);
    }
}

bool gstpipeline::ready() const
{
    return pipeline && playing;
}

void gstpipeline::buffercallback(void *user_data, void *buffer)
{
    void *Buffer = (GstBuffer *)buffer;
    gstpipeline *_pipeline = static_cast<gstpipeline *>(user_data);
    _pipeline->GstBufferCallBack(buffer);
}

void gstpipeline::GstBufferCallBack(void *buffer)
{
    if(buffer != NULL)
    {
        if(stop_push)
        {
             gst_buffer_unref((GstBuffer *)buffer);
        }
        else
        {
            pthread_mutex_lock(&sync_lock);
            sync_params.curr_buffer = (GstBuffer *) buffer;
            bufferqueue.push(sync_params);

            /* first buffer or if there is no data in queue then push from here */
            if(oldBuffer == NULL || nodata_to_push == true)
            {
                currentBuffer = bufferqueue.front().curr_buffer;
                bufferqueue.pop();
                push_appsrc(currentBuffer);
                nodata_to_push = false;
            }
            pthread_mutex_unlock(&sync_lock);
        }
    }
}

void gstpipeline::push_appsrc(GstBuffer *buffer)
{
    GstFlowReturn ret;

    if(oldBuffer == NULL)
    {
        ret = gst_app_src_push_buffer(GST_APP_SRC(appsrc), gst_buffer_ref(buffer));
        if(ret !=  GST_FLOW_OK){
            std::cout << "push buffer returned  "<<ret<< std::endl;
            gst_buffer_unref(buffer);
        }
    }
    else if(oldBuffer == buffer)
    {
        ret = gst_app_src_push_buffer(GST_APP_SRC(appsrc), buffer);
        if(ret !=  GST_FLOW_OK){
            std::cout << "push buffer returned  "<<ret<< std::endl;
            gst_buffer_unref(buffer);
        }
    }
    else
    {
        ret = gst_app_src_push_buffer(GST_APP_SRC(appsrc), gst_buffer_ref(buffer));
        gst_buffer_unref(oldBuffer);
        if(ret !=  GST_FLOW_OK){
            std::cout << "push buffer returned  "<<ret<< std::endl;
            gst_buffer_unref(buffer);
        }
    }

    if(ret ==  GST_FLOW_OK)
        oldBuffer = buffer;
}

gboolean gstpipeline::on_buffer_probe (GstPad *pad, GstMiniObject *data, gpointer user_data)
{
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    printf("%s,%lld.%.9ld\n", (const char *) user_data, (long long) ts.tv_sec, ts.tv_nsec);

    return TRUE;
}

void gstpipeline::requestTimingProbe(GstElement *element, const char *inTag, const char *outTag)
{
    if (getenv("OSG_STREAM_PROFILE_CSC") != NULL)
    {
        GstPad *pad = gst_element_get_static_pad (element, "sink");

#if GST_CHECK_VERSION(1,0,0)
        gst_pad_add_probe(pad,GST_PAD_PROBE_TYPE_BUFFER,
                    GstPadProbeCallback(gstpipeline::on_buffer_probe), (gpointer)inTag,NULL);
        gst_object_unref (pad);
        pad = gst_element_get_static_pad (element, "src");
        gst_pad_add_probe(pad,GST_PAD_PROBE_TYPE_BUFFER,
                    GstPadProbeCallback(gstpipeline::on_buffer_probe), (gpointer)outTag,NULL);
        gst_object_unref (pad);
#else
        gst_pad_add_buffer_probe(pad, G_CALLBACK (gstpipeline::on_buffer_probe), (gpointer)inTag);
        gst_object_unref (pad);
        pad = gst_element_get_static_pad (element, "src");
        gst_pad_add_buffer_probe(pad, G_CALLBACK(gstpipeline::on_buffer_probe), (gpointer)outTag);
        gst_object_unref (pad);
#endif
    }
}

void gstpipeline::play()
{
    std::cout<<"func "<<__func__<<" setting pipline state to play "<<std::endl;
    stop_push = false;
    gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_PLAYING);
}

void gstpipeline::stop()
{
    std::cout<<"func "<<__func__<<" setting pipline state to NULL "<<std::endl;
    stop_push = true;
    gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_NULL);
}

void gstpipeline::close()
{
    std::cout<<"func "<<__func__<<"closing pipline "<<std::endl;
    if (pipeline)
    {
        stop();
        gst_element_set_state (pipeline, GST_STATE_NULL);
        gst_buffer_unref (oldBuffer);

//        g_signal_handler_disconnect(bus, callbackId);
        gst_object_unref (bus);

        gst_object_unref(appsrc);
        gst_object_unref(GST_OBJECT(pipeline));

        g_main_loop_unref (loop);
        g_main_loop_quit (loop);
        g_main_context_pop_thread_default(context);

        pthread_join(thread, NULL);
    }
}
} // osgGstpipeline
